import { type NextRequest, NextResponse } from "next/server"
import { gameStore } from "@/lib/game-store"

export async function GET(request: NextRequest, { params }: { params: { gameId: string } }) {
  const gameId = params.gameId
  let game = gameStore.getGame(gameId)

  if (!game) {
    game = gameStore.createGame(gameId)
  }

  return NextResponse.json(game)
}

export async function POST(request: NextRequest, { params }: { params: { gameId: string } }) {
  const gameId = params.gameId
  const body = await request.json()
  const { action, payload } = body

  const game = gameStore.updateGame(gameId, (currentGame) => {
    switch (action) {
      case "ADD_PLAYER":
        const existingPlayer = currentGame.players.find((p) => p.name === payload.name)
        if (existingPlayer) {
          return currentGame // Player already exists
        }

        return {
          ...currentGame,
          players: [
            ...currentGame.players,
            {
              id: payload.id,
              name: payload.name,
              bet: null,
              hasRevealed: false,
              score: 0,
              joinedAt: Date.now(),
            },
          ],
        }

      case "PLACE_BET":
        return {
          ...currentGame,
          players: currentGame.players.map((p) =>
            p.id === payload.playerId ? { ...p, bet: payload.bet, hasRevealed: true } : p,
          ),
        }

      case "START_BETTING":
        return { ...currentGame, phase: "betting" }

      case "REVEAL_BETS":
        return { ...currentGame, phase: "revealed" }

      case "FINISH_GAME":
        // Calculate winner
        let winner = null
        let closestDiff = Number.POSITIVE_INFINITY

        currentGame.players.forEach((player) => {
          if (player.bet !== null) {
            const diff = Math.abs(player.bet - payload.actualTime)
            if (diff < closestDiff) {
              closestDiff = diff
              winner = player.name
            }
          }
        })

        return {
          ...currentGame,
          phase: "finished",
          actualTime: payload.actualTime,
          winner,
          players: currentGame.players.map((p) => ({
            ...p,
            score: p.bet !== null && p.name === winner ? p.score + 1 : p.score,
          })),
        }

      case "RESET_GAME":
        return {
          ...currentGame,
          phase: "joining",
          actualTime: null,
          winner: null,
          players: currentGame.players.map((p) => ({
            ...p,
            bet: null,
            hasRevealed: false,
          })),
        }

      default:
        return currentGame
    }
  })

  if (!game) {
    return NextResponse.json({ error: "Game not found" }, { status: 404 })
  }

  return NextResponse.json(game)
}
